/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "numpad.h"
#include "miscellaneous.h"
#include <QString>

/* Constructor; Configure the display sources, button source and set up the blink timer for the cursor blinking */
Numpad::Numpad(int dec_length, int fract_length, QLabel *display, QLabel *unit_display, QPushButton *period_btn, QPushButton *sign_btn)
	: decimal_length(dec_length),
	  fractal_length(fract_length),
	  displaylabel(display),
	  unitlabel(unit_display),
	  period_button(period_btn),
	  sign_button(sign_btn)
{
	blink_timer = new QTimer(this);
	connect(blink_timer, &QTimer::timeout, this, &Numpad::cursor_blink_update);
}

/* Configure the pointer to the unit display */
void Numpad::setUnitDisplay(QLabel *unit_display)
{
	unitlabel = unit_display;
}

/* Configure the pointer to the digit display */
void Numpad::setDisplay(QLabel *display)
{
	displaylabel = display;
}

/* Configure the pointer to the source button that provides the number */
void Numpad::setOutputButton(QPushButton *button)
{
	target_button = button;
}

void Numpad::setSignButton(QPushButton *button)
{
	sign_button = button;
}

/* Loads the parameters into the numpad and prepares everything for use */
void Numpad::load_value(double *parameter, QString unit, bool enable_period, bool enable_sign, double scale_multiplier)
{
	overwrite_permission = true;

	number_ptr = parameter;
	multiplier = scale_multiplier;

	unitlabel->setText(unit);
	period_button->setEnabled(enable_period);
	sign_button->setVisible(enable_sign);

	if (*number_ptr < 0)
	{
		set_number_sign(NumpadSign::Negative);
	}
	else
	{
		set_number_sign(NumpadSign::Positive);
	}

	numberstring = zeroChopper(QString::number(abs(*parameter * multiplier), 'f', fractal_length));
	set_index_cursor_relative_period(numberstring.length() - get_index_period(numberstring) - 1);		//set the index position to the right corner.

	update_display();
	blink_timer->start(500);
}

/* Updates the display label with user inputs */
void Numpad::update_display()
{
	if(numberstring == "")
	{
		numberstring = "0";
	}

	//Reset the numberstring to a clean unpadded state.
	numberstring = QString::number(numberstring.toDouble(), 'f', calculate_precision(numberstring));
	if(calculate_precision(numberstring) == 0 && get_index_cursor_relative_period() >= 0 )
	{
		numberstring.insert(get_index_period(numberstring), ".");
	}

	pad_with_zeroes();
	update_cursor_index();

	QString temp = numberstring;
	if(cursor_status == true)
	{
		displaylabel->setText(temp.insert(get_index_cursor(), "<u>"));
		displaylabel->setText(temp.insert(get_index_cursor()+4, "</u>"));		//+4 to account for the newly added <u>
	}
	else
	{
		displaylabel->setText(numberstring);
		cursor_status = false;
	}
}

/* Save the user input, return numpad to a default state and emit a signal that the value has been confirmed */
void Numpad::press_confirm_input()
{
	if (target_button == NULL)
	{
		return;
	}

	blink_timer->stop();
	period_button->setEnabled(true);

	*number_ptr = (number_sign + numberstring).toDouble() / multiplier;
	emit value_confirmed_signal(target_button, number_ptr);
	target_button = NULL;
	displaylabel->clear();
	unitlabel->clear();
}

/* Delete a single input */
void Numpad::press_delete()
{
	overwrite_permission = false;
	numberstring.chop(1);

	if (numberstring == "")
	{
		numberstring += "0";
	}

	//set the index position to the right corner.
	set_index_cursor_relative_period(numberstring.length() - get_index_period(numberstring) - 1);
	update_display();
}

/* Reset the display to 0 */
void Numpad::press_delete_all()
{
	numberstring = "0";
	set_index_cursor_relative_period(-1);
	update_display();
}

/* Insert a period, if applicable */
void Numpad::press_period()
{
	if (overwrite_permission == true)
	{
		overwrite_permission = false;
		press_delete_all();
	}

	if (numberstring.contains("."))
	{
		return;
	}

	numberstring += ".";
	set_index_cursor_relative_period(0);
	update_display();
}

/* insert numbers */
void Numpad::press_1()
{
	add_number(1);
}

void Numpad::press_2()
{
	add_number(2);
}

void Numpad::press_3()
{
	add_number(3);
}

void Numpad::press_4()
{
	add_number(4);
}

void Numpad::press_5()
{
	add_number(5);
}

void Numpad::press_6()
{
	add_number(6);
}

void Numpad::press_7()
{
	add_number(7);
}

void Numpad::press_8()
{
	add_number(8);
}

void Numpad::press_9()
{
	add_number(9);
}

void Numpad::press_0()
{
	add_number(0);
}

/* Handle number insertion if applicable and move cursor to rightmost position */
void Numpad::add_number(int num)
{
	if (overwrite_permission == true)
	{
		overwrite_permission = false;
		press_delete_all();
	}

	if (get_input_permission() == false)
	{
		return;		//enforce the maximum character count
	}

	if (numberstring == "0")
	{
		numberstring = "";
	}

	numberstring += QString::number(num);

	//set the index position to the right corner.
	set_index_cursor_relative_period(numberstring.length() - get_index_period(numberstring) - 1);
	update_display();
}

/**********************************************************************************************************************************************************************************
 * Numpad indicator management
 * *******************************************************************************************************************************************************************************/
/* Change the sign of the number */
void Numpad::press_sign()
{
	if (sign_button->text() == "+")
	{
		set_number_sign(NumpadSign::Negative);
	}
	else if (sign_button->text() == "-")
	{
		set_number_sign(NumpadSign::Positive);
	}
}

void Numpad::set_number_sign(Numpad::NumpadSign sign)
{
	if (sign == NumpadSign::Negative)
	{
		number_sign = '-';
	}
	else if (sign == NumpadSign::Positive)
	{
		number_sign = '+';
	}

	sign_button->setText(QString(number_sign));
}

/* Increment value at cursor */
void Numpad::press_plus()
{
	increment_number(+1);
}

/* Decrement value at cursor */
void Numpad::press_minus()
{
	increment_number(-1);
}

/* Handle decrementation/incrementation - ensure values stay within supported limits, etc */
void Numpad::increment_number(int incrementation)
{
	overwrite_permission = false;
	if (get_index_cursor_relative_period() == 0)
	{
		return;		//Don't do anything if the cursor is targetting the period.
	}
	
	double	temp_number = numberstring.toDouble();
	temp_number += incrementation * pow(10, get_cursor_magnitude());

	if (get_increment_permission(temp_number) == false)
	{
		return;		//Don't do anything if the value exceeds the minimum/maximum supported input values.
	}

	numberstring = QString::number(temp_number, 'f', calculate_precision(QString::number(temp_number,'f', fractal_length+1)));
	update_display();
}

/* Move cursor left, clear period if nothing after period */
void Numpad::press_indicator_left()
{
	overwrite_permission = false;
	if (get_cursor_magnitude() + 1 >= decimal_length)
	{
		return;		//enforce maximum number length
	}

	set_index_cursor_relative_period(get_index_cursor_relative_period() - 1);

	if (get_index_cursor_relative_period() == 0 && numberstring.endsWith(".0"))	//Make the indicator sit on the period properly...
	{
		press_delete();
		press_period();
	}

	update_display();
	force_cursor_highlight(true);
}

/* Move cursor right, add period (if applicable) if the moving beyond the rightmost character. */
void Numpad::press_indicator_right()
{
	overwrite_permission = false;
	if (period_button->isEnabled() == false && get_cursor_magnitude() == 0)
	{
		return;		//Do not go into fractal territory if period button is disabled
	}
	if (get_cursor_magnitude() <= -fractal_length)
	{
		return;		//enforce maximum number length
	}

	if (get_index_cursor_relative_period() == -1 && !numberstring.contains("."))
	{
		press_period();
	}
	else
	{
		set_index_cursor_relative_period(get_index_cursor_relative_period() + 1);
	}

	update_display();
	force_cursor_highlight(true);
}

/* Pad the empty space between cursor and actual number with zeroes. */
void Numpad::pad_with_zeroes()
{
	if (get_index_cursor_relative_period() < 0)
	{
		int padding = -get_index_period(numberstring) - get_index_cursor_relative_period();
		for (int i = 0; i < padding; i++)
		{
			numberstring.insert(0, '0');
		}
	}
	else if (get_index_cursor_relative_period() > 0)
	{
		int padding = get_index_cursor_relative_period() - calculate_precision(numberstring);
		for (int i = 0; i < padding; i++)
		{
			numberstring += "0";
		}
	}
}


/**********************************************************************************************************************************************************************************
 * Get/Set indexing values
 * *******************************************************************************************************************************************************************************/
/* Return the index position of the period. If there is no period, attach it to the back of the string and return that. */
int Numpad::get_index_period(QString numstring)
{
	QString temp_string = numstring;
	if (!temp_string.contains("."))			//if number does not contain a period add one at the end.
	{
		temp_string += ".";
	}
	return temp_string.indexOf(".");
}

/* Return the position of the cursor relatively to the string length */
int Numpad::get_index_cursor()
{
	return index_cursor;
}

/* Return the cursor index relative to the period position */
int Numpad::get_index_cursor_relative_period()
{
	return index_cursor_relative_period;
}

/* Set cursor index relative to the period position */
void Numpad::set_index_cursor_relative_period(int val)
{
	index_cursor_relative_period = val;
}

/* Synchronize the real cursor index position, with the cursor index position relative to the period. */
void Numpad::update_cursor_index()
{
	index_cursor = get_index_cursor_relative_period() + get_index_period(numberstring);
}

/* Return the magnitude of the selected index position relative to the period. Meaningless if index is on top of period. */
int Numpad::get_cursor_magnitude()
{
	magnitude_cursor = -get_index_cursor_relative_period();
	if (get_index_cursor_relative_period() < 0)
	{
		magnitude_cursor -= 1;
	}
	return magnitude_cursor;
}


/**********************************************************************************************************************************************************************************
 * Miscellaneous
 * *******************************************************************************************************************************************************************************/
/* Return the minimum precision necessary to display the number without loss of information. */
int Numpad::calculate_precision(QString numstring)
{
	QString temp_string = zeroChopper(numstring);
	int current_precision = temp_string.length() - get_index_period(temp_string) - 1 ;

	if (current_precision < 0 || numstring.endsWith("."))
	{
		current_precision = 0;
	}
	return current_precision;
}

/* Do the cursor blinking Animation */
void Numpad::cursor_blink_update()
{
	QString temp = numberstring;
	if(cursor_status == false)
	{
		displaylabel->setText(temp.insert(get_index_cursor(), "<u>"));
		displaylabel->setText(temp.insert(get_index_cursor()+4, "</u>"));	//+4 to account for the newly added <u>
	}
	else
	{
		displaylabel->setText(numberstring);
	}

	cursor_status = !cursor_status;
}

/* Force the cursor blinking animation into a specific state */
void Numpad::force_cursor_highlight(bool state)
{
	QString temp = numberstring;
	if(state == true)
	{
		displaylabel->setText(temp.insert(get_index_cursor(), "<u>"));
		displaylabel->setText(temp.insert(get_index_cursor()+4, "</u>"));	//+4 to account for the newly added <u>
	}
	else
	{
		displaylabel->setText(numberstring);
	}

	cursor_status = !state;
	blink_timer->start(500);		//reset blink_timer
}

/* Check if incrementing will cause the value to exceed input limits of the numpad. */
bool Numpad::get_increment_permission(double val)
{
	if (val < 0)
	{
		return false;
	}
	if (val > pow(10,decimal_length) - pow(10, -fractal_length))
	{
		return false;
	}
	return true;
}

/* Check if numpad still has space left to fill with inputs. */
bool Numpad::get_input_permission()
{
	if (numberstring.contains('.'))
	{
		if(numberstring.length() - get_index_period(numberstring) - 1 >= fractal_length)
		{
			return false;
		}
	}
	else
	{
		if (get_index_period(numberstring) >= decimal_length)
		{
			return false;
		}
	}
	return true;
}
